<?php

namespace Comitatus\Controller;

use BaseModule\Service\Config\Config;
use Comitatus\Filter\ProfileFilter;
use Comitatus\Filter\UserSettingsFilter;
use Comitatus\Form\ProfileForm;
use Comitatus\Form\UserSettingsForm;
use Comitatus\Service\Database\DatabaseService;
use Comitatus\Service\Password\PasswordService;
use Zend\Db\Adapter\Adapter;
use Zend\Mvc\MvcEvent;
use Zend\Validator\EmailAddress;
use Zend\View\Model\JsonModel;

class ProfileController extends BaseController
{
    
    public function indexAction()
    {
        
        $id           = $this->params('param1', '');
        $user         = [];
        $regularOrder = [];
        $preOrder     = [];
        
        // if id is empty get the ID of current user;
        if ($id == '')
            $id = $this->currentUser['id'];
        
        $systemSettings        = $this->databaseService->selectData(DatabaseService::APP_SETTINGS_TBL);
        $isRewardSystemEnabled = $systemSettings->count() ? $systemSettings->current()['reward_system'] : 0;
        
        $paymentSchemes = $this->databaseService->selectData(DatabaseService::PAYMENT_SCHEME_TBL, ['is_archive' => 0]);
        
        $schoolYearInfo = $this->databaseService->selectData(DatabaseService::YR_SCHEDULE_TBL, ['is_active' => 1])->current();
        $allSchoolYear  = $this->databaseService->selectData(DatabaseService::YR_SCHEDULE_TBL, [], [], '', [], false, 10);
        $levels         = $this->databaseService->selectData(DatabaseService::LEVEL_TBL, ['is_archive' => 0], ['name ASC']);
        $sections       = [];
        
        if (!empty($id)) {
            $user = $this->accountManager->getAllUserInfo($id);
            
            // get sections
            if (isset($user['basic'])) {
                if (isset($user['basic']['level']) && $user['basic']['level'] != '') {
                    $sections = $this->databaseService->selectData(DatabaseService::SECTION_TBL, ['is_archive' => 0]);
                }
            }
            
        }
        
        //        echo '<pre>';
        //        print_r($user);
        //        exit;
        
        // segregate orders
        if (isset($user['canteen']) && count($user['canteen'])) {
            foreach ($user['canteen'] as $item => $product) {
                if ($product['is_preorder'])
                    $preOrder[] = $product;
                else $regularOrder[] = $product;
            }
        }
        
        return [
            'access_right'          => $this->accountManager->getAccessRight($id),
            'tabs'                  => $this->config['allowed-tabs'],
            'userInformation'       => $user,
            'isAdmin'               => $this->currentUser['is_admin'],
            'isRewardSystemEnabled' => $isRewardSystemEnabled,
            'paymentSchemes'        => $paymentSchemes,
            'regularOrders'         => $regularOrder,
            'preOrders'             => $preOrder,
            'schoolYear'            => $schoolYearInfo,
            'allSchoolYear'         => $allSchoolYear,
            'levels'                => $levels,
            'sections'              => $sections,
        ];
    }
    
    public function addFormAction()
    {
        $paymentSchemes = $this->databaseService->selectData(DatabaseService::PAYMENT_SCHEME_TBL, ['is_archive' => 0]);
        $levels         = $this->databaseService->selectData(DatabaseService::LEVEL_TBL, ['is_archive' => 0]);
        
        if (strtolower($this->getRequest()->getMethod()) == 'post') {
            $data = $this->params()->fromPost('data', []);
            
            $validation = $this->validateProfileForm($data);
            if ($validation['status'] == 0) {
                return new JsonModel([
                    'status'  => 0,
                    'message' => 'Validation failed! Check the error message(s) and try again.',
                    'errors'  => $validation['message'],
                ]);
            }
            
            $result = $this->saveUserInformation($data);
            return new JsonModel($result);
            
        }
        
        return [
            'levels'         => $levels,
            'paymentSchemes' => $paymentSchemes,
        ];
    }
    
    private function saveUserInformation($data)
    {
        /** @var PasswordService $passwordManager */
        $passwordManager = $this->serviceLocator->get(PasswordService::class);
        $static          = $this->config['static'];
        $dateNow         = date('Y-m-d');
        
        $imageSource   = isset($data['img_src']) ? $data['img_src'] : '';
        $imageFileName = isset($data['file_name']) ? $data['file_name'] : '';
        $isUpload      = isset($data['is_upload']) ? $data['is_upload'] : 0;
        $saveOptions   = isset($data['saveOptions']) ? $data['saveOptions'] : 0;
        $paymentScheme = isset($data['payment_scheme']) && $data['payment_scheme'] != '' ? $data['payment_scheme'] : NULL;
        
        // formatting data for database data integrity
        $data['access_permit'] = $data['role'];
        $data['birthday']      = date('Y-m-d', strtotime($data['birthday']));
        $data['card_id']       = $data['card_id'] != '' ? $data['card_id'] : NULL;
        $data['postal']        = $data['postal'] != '' ? $data['postal'] : NULL;
        $data['postal']        = $data['postal'] != '' ? $data['postal'] : NULL;
        $data['term']          = $data['term'] != '' ? $data['term'] : NULL;
        $data['section']       = $data['section'] != '' ? $data['section'] : 0;
        $data['level']         = $data['level'] != '' ? $data['level'] : 0;
        $data['modified_by']   = $this->userModified;
        
        // unset post data that is not needed on the database insertion
        unset($data['img_src'], $data['file_name'], $data['is_upload'], $data['saveOptions'], $data['role'], $data['payment_scheme']);
        
        //If banner doesn't exist,
        if ($data['id'] == '') {
            $data['date_created'] = date('Y-m-d H:i:s');
            unset($data['id']);
            
            // generate password
            $generatedPassword = $passwordManager->generateStrongPassword(8, false, 'luds');
            $data['password']  = password_hash($generatedPassword, PASSWORD_BCRYPT);
            $data['temp_key']  = $generatedPassword;
            
            if (!empty($data['email']))
                $data['access_token'] = password_hash($data['email'], PASSWORD_BCRYPT);
            else $data['access_token'] = password_hash(uniqid(), PASSWORD_BCRYPT);
            
            // get default app settings
            $theme       = 'skin-red';
            $schoolId    = 0;
            $appSettings = $this->databaseService->selectData(DatabaseService::APP_SETTINGS_TBL);
            if ($appSettings->count()) {
                $theme    = $appSettings->current()['default_theme'];
                $schoolId = $appSettings->current()['school_id'];
                $schoolId = $schoolId > 10 ? '0' . $schoolId : $schoolId;
            }
            
            // insert user data
            $lastId = $this->databaseService->createData(DatabaseService::USER_TABLE, $data);
            if (intval($lastId)) {
                // check role
                if ($data['access_permit'] == 2)
                    $type = 'S'; //student
                else $type = 'E'; //Employee
                
                //check if its new year
                //if new year start ID count to 1
                $isNewYear     = 0;
                $yearNow       = date('Y');
                $thisYearUsers = $this->databaseService->selectData(DatabaseService::USER_TABLE, "user_id LIKE '%$yearNow%'");
                $recordCount   = $thisYearUsers->count();
                if (!$recordCount)
                    $isNewYear = 1;
                
                // generate userId
                $generatedId = $this->accountManager->generateId($type, $schoolId, $recordCount, 5, $isNewYear);
                
                // upload profile picture
                $generatedPhotoName = '';
                if (intval($isUpload) == 1) {
                    $saveImage          = $this->imageManager->saveBase64Image($imageSource, $generatedId, $static . '/users');
                    $generatedPhotoName = $saveImage['filename'];
                }
                
                // update users table for additional information
                $this->databaseService->updateData(DatabaseService::USER_TABLE, ['id' => $lastId, 'user_id' => $generatedId, 'picture' => $generatedPhotoName]);
                
                // create user settings and wallet
                $this->databaseService->createData(DatabaseService::USER_SETTINGS_TBL, [
                    'user_id'             => $lastId,
                    'theme'               => $theme,
                    'max_daily_spending'  => 100,
                    'notification_charge' => 1,
                    'payment_scheme'      => $paymentScheme,
                    'date_created'        => date('Y-m-d H:i:s'),
                    'modified_by'         => $this->userModified
                ]);
                $this->databaseService->createData(DatabaseService::USER_WALLET_TBL, [
                    'user_id'          => $lastId,
                    'date_created'     => date('Y-m-d H:i:s'),
                    'prepaid_balance'  => 0,
                    'postpaid_balance' => 0,
                    'modified_by'      => $this->userModified
                ]);
            }
            
            // returns
            $return['_guid']   = $lastId;
            $return['status']  = 1;
            $return['message'] = 'User Added Successfully!';
            
        } else {
            // updating user information
            $this->databaseService->updateData(DatabaseService::USER_TABLE, $data);
            $return['status']  = 1;
            $return['message'] = 'Profile Updated Successfully!';
        }
        
        $return['save_option'] = $saveOptions;
        
        return $return;
    }
    
    private function validateProfileForm($data)
    {
        if (isset($data['id']) && $data['id'] != '')
            $options = ['user_id' => $data['id']];
        else $options = ['user_id' => 0];
        
        $form   = new ProfileForm($data);
        $filter = new ProfileFilter($this->serviceLocator, $options);
        
        $form->setInputFilter($filter);
        $form->setData($data);
        
        if ($form->isValid()) {
            $return['status']  = 1;
            $return['message'] = [];
        } else {
            $return['status']  = 0;
            $return['message'] = $filter->getMessages();
        }
        
        return $return;
    }
    
    public function getGuardianByUserIdAction()
    {
        $userId    = $this->params('param1');
        $guardians = [];
        
        if (isset($userId) && $userId != '') {
            $guardians = $this->accountManager->getUserGuardians($userId);
        }
        
        return new JsonModel(['result' => $guardians]);
    }
    
    public function saveGuardianAction()
    {
        $errors    = [];
        $guardians = $this->params()->fromPost('data', []);
        $user      = $this->params()->fromPost('user');
        $dataCount = count($guardians);
        
        if ($dataCount) {
            foreach ($guardians as $key => $value) {
                $imgSrc   = $value['img_src'];
                $fileName = $value['file_name'];
                $isUpload = $value['is_upload'];
                unset($value['img_src'], $value['file_name'], $value['is_upload']);
                
                if (empty($value['first_name'])) {
                    unset($guardians[$key]);
                } else {
                    
                    if ($value['email']) {
                        $emailValidator = new EmailAddress();
                        if (!$emailValidator->isValid($value['email']))
                            $errors[$key] = $emailValidator->getMessages();
                    }
                    
                    if (!count($errors)) {
                        $value['modified_by'] = $this->userModified;
                        $value['card_load']   = 0;
                        
                        if ($isUpload) {
                            $saveImage = $this->imageManager->saveBase64Image($imgSrc, 'g0' . ($key + 1) . '_' . $user, $this->config['static'] . '/users/guardians');
                            if ($saveImage)
                                $value['picture'] = $saveImage['filename'];
                        }
                        
                        if (intval($value['id'])) {
                            $this->databaseService->updateData(DatabaseService::GUARDIAN_TBL, $value);
                        } else {
                            unset($value['id']);
                            $value['date_created'] = date('Y-m-d H:i:s');
                            $this->databaseService->createData(DatabaseService::GUARDIAN_TBL, $value);
                        }
                    }
                    
                }
            }
            
            if (count($errors))
                return new JsonModel(['status' => 0, 'message' => 'Validation Failed!', 'errors' => $errors]);
            else return new JsonModel(['status' => 1, 'message' => 'Information updated!']);
        }
        
    }
    
    public function updateSkinAction()
    {
        $skin       = $this->params()->fromPost('skin', 'skin-red');
        $userId     = $this->currentUser['id'];
        $isAdmin    = $this->currentUser['is_admin'];
        $db         = $this->sessionContainer->offsetGet('school_db');
        $modifiedBy = $this->userModified;
        
        // point to school
        if (isset($this->config['db']['adapters']['db_' . $db])) {
            $adapter = $this->serviceLocator->get('db_' . $db);
            $this->databaseService->setDatabase($adapter);
        }
        
        // point to admin database
        if ($isAdmin) {
            $modifiedBy = $userId;
            $adapter    = $this->serviceLocator->get('main');
            $this->databaseService->setDatabase($adapter);
        }
        
        if ($userId != '') {
            // execute update
            $skinData = array('user_id' => $userId, 'theme' => $skin, 'modified_by' => $modifiedBy);
            $update   = $this->databaseService->updateData(DatabaseService::USER_SETTINGS_TBL, $skinData, 'user_id');
            if ($update->valid()) {
                $this->sessionContainer->offsetGet('current_user')['theme'] = $skin;
                return new JsonModel(['status' => 1]);
            } else return new JsonModel(['status' => 0]);
        }
        
        return new JsonModel(['status' => 0, 'msg' => 'invalid request!']);
    }
    
    public function saveAppSettingsAction()
    {
        if (strtolower($this->getRequest()->getMethod()) == 'post') {
            $return = [];
            $data   = $this->params()->fromPost('data', []);
            
            
            $validation = $this->validateAppSettings($data);
            if ($validation['status'] == 0) {
                return new JsonModel([
                    'status'  => 0,
                    'message' => 'Validation failed! Check the error message(s) and try again.',
                    'errors'  => $validation['message'],
                ]);
            }
            
            $data['modified_by'] = $this->userModified;
            if ($data['e_wallet_pin'] != '') {
                $data['e_wallet_pin'] = password_hash($data['e_wallet_pin'], PASSWORD_BCRYPT);
            }
            
            if (intval($data['user_id'])) {
                $update           = $this->databaseService->updateData(DatabaseService::USER_SETTINGS_TBL, $data, 'user_id');
                $return['status'] = $update->valid() ? 1 : 0;
            } else {
                $value['date_created'] = date('Y-m-d H:i:s');
                $last_id               = $this->databaseService->createData(DatabaseService::USER_SETTINGS_TBL, $data);
                $return['status']      = intval($last_id) ? 1 : 0;
            }
            
            
            if ($return['status'] == 1) {
                return new JsonModel(['status' => 1, 'message' => 'Settings Updated Successfully!']);
            } else return new JsonModel(['status' => 0, 'message' => 'Something went wrong. Please try again!']);
            
        } else return new JsonModel(['message' => 'Invalid request!']);
    }
    
    private function validateAppSettings($data)
    {
        $form   = new UserSettingsForm($data);
        $filter = new UserSettingsFilter($this->serviceLocator);
        
        $form->setInputFilter($filter);
        $form->setData($data);
        
        if ($form->isValid()) {
            $return['status']  = 1;
            $return['message'] = [];
        } else {
            $return['status']  = 0;
            $return['message'] = $filter->getMessages();
        }
        
        return $return;
    }
}