<?php

namespace Comitatus\Service\Account;

use BaseModule\Service\AbstractBaseService;
use Comitatus\Service\Database\DatabaseService;
use Zend\Db\Sql\Sql;
use Zend\ServiceManager\ServiceLocatorInterface;

class AccountManager extends AbstractBaseService
{
    const ACCESS_DEV                = 0b00000010;
    const ACCESS_GUEST              = 0b00000000;
    const ACCESS_ADMIN              = 0b00000001;
    const ACCESS_STUDENT            = 0b00000010;
    const ACCESS_ACCOUNTING_MANAGER = '4a';
    const ACCESS_CANTEEN_MANAGER    = '4b';
    const ACCESS_SECURITY           = '4e';
    const ACCESS_TEACHER            = '4f';
    const ACCESS_REGISTRAR          = '4g';
    const ACCESS_REGULAR_EMPLOYEE   = '4h';
    const ACCESS_ACCOUNTING_CANTEEN = 0b00000101;
    
    public function getAccessRight($accessPermit = self::ACCESS_GUEST)
    {
        switch ($accessPermit) {
            case static::ACCESS_ADMIN:
                return 'all';
                break;
            case static::ACCESS_STUDENT:
                return 'student';
                break;
            case static::ACCESS_ACCOUNTING_MANAGER:
                return 'accounting-manager';
                break;
            case static::ACCESS_CANTEEN_MANAGER:
                return 'canteen-manager';
                break;
            case static::ACCESS_SECURITY:
                return 'security';
                break;
            case static::ACCESS_TEACHER:
                return 'teacher';
                break;
            case static::ACCESS_REGISTRAR:
                return 'registrar';
                break;
            case static::ACCESS_ACCOUNTING_CANTEEN:
                return 'accounting-canteen';
                break;
            default:
                return 'employee';
                break;
        }
    }
    
    public function searchUser($search, $columns = [])
    {

        $select = $this->sql->select(DatabaseService::USER_TABLE);
        $select->join(DatabaseService::LEVEL_TBL, DatabaseService::LEVEL_TBL . '.id = ' . DatabaseService::USER_TABLE . '.level', ['level_name' => 'name'], $select::JOIN_LEFT);
        $select->join(DatabaseService::SECTION_TBL, DatabaseService::SECTION_TBL . '.id = ' . DatabaseService::USER_TABLE . '.section', ['section_name' => 'name'], $select::JOIN_LEFT);

        if (!empty($search)) {
            foreach ($columns as $key => $column) {
                $operator = $key == 0 ? 'and' : 'or';
                $select->where->$operator->like(DatabaseService::USER_TABLE . '.' . $column, '%' . $search . '%');
            }
        }

        $result = $this->sql->prepareStatementForSqlObject($select)->execute();

        $array = [];
        if ($result->count()) {
            foreach ($result as $key => $value) {
                $array[$key] = $value;
            }
        }

        return $array;
    }

    public function listAllUser($where, $type = 'a')
    {
        $select = $this->sql->select(DatabaseService::USER_TABLE);
        $select->where($where);

        if ($type == 's')
            $select->where->equalTo('access_permit', 2);
        if ($type == 't')
            $select->where->notEqualTo('access_permit', 2);

        return $this->sql->prepareStatementForSqlObject($select)->execute();
    }

    public function getAllStudents()
    {
        $select = $this->sql->select(DatabaseService::USER_TABLE);
        $select->where(['access_permit' => 2, 'account_close' => 0]);
        return $this->sql->prepareStatementForSqlObject($select)->execute();
    }

    public function getAllStaff()
    {
        $select = $this->sql->select(DatabaseService::USER_TABLE);
        $select->where
            ->equalTo('account_close', 0)
            ->and
            ->notEqualTo('access_permit', 2);
        return $this->sql->prepareStatementForSqlObject($select)->execute();
    }

    public function getAllGuardians()
    {
        $select = $this->sql->select(DatabaseService::GUARDIAN_TBL);
        return $this->sql->prepareStatementForSqlObject($select)->execute();
    }

    public function getUserInfo($idOrEmail)
    {
        $select = $this->sql->select(DatabaseService::USER_TABLE);
        $select->join(DatabaseService::LEVEL_TBL, DatabaseService::LEVEL_TBL . '.id = ' . DatabaseService::USER_TABLE . '.level', ['level_name' => 'name'], $select::JOIN_LEFT);
        $select->join(DatabaseService::SECTION_TBL, DatabaseService::SECTION_TBL . '.id = ' . DatabaseService::USER_TABLE . '.section', ['section_name' => 'name'], $select::JOIN_LEFT);
        $select
            ->where
            ->equalTo(DatabaseService::USER_TABLE . '.id', $idOrEmail)
            ->or
            ->equalTo(DatabaseService::USER_TABLE . '.email', $idOrEmail);
        $result = $this->sql->prepareStatementForSqlObject($select)->execute();

        if ($result->count()) {
            $current = $result->current();
            unset($current['password'], $current['access_token'], $current['reset_pw_token'], $current['temp_key']);
            return $current;
        }

        return [];
    }

    public function getUserCanteenInfo($id)
    {
        $select = $this->sql->select(DatabaseService::CANTEEN_SOLD_TBL);
        $select->join(DatabaseService::CANTEEN_TBL, DatabaseService::CANTEEN_TBL . '.id = ' . DatabaseService::CANTEEN_SOLD_TBL . '.product_id', ['product_name' => 'name', 'product_price' => 'price'], $select::JOIN_LEFT);

        $select
            ->where
            ->equalTo(DatabaseService::CANTEEN_SOLD_TBL . '.user_id', $id);
        $select->order('id DESC');

        $result = $this->sql->prepareStatementForSqlObject($select)->execute();

        $array = [];
        if ($result->count()) {
            foreach ($result as $key => $value) {
                $array[$key] = $value;
            }
        }

        return $array;
    }

    public function getUserWallet($id)
    {
        $select = $this->sql->select(DatabaseService::USER_WALLET_TBL);
        $select->join(DatabaseService::REWARDS_TABLE, DatabaseService::REWARDS_TABLE . '.user_id = ' . DatabaseService::USER_WALLET_TBL . '.user_id', ['points'], $select::JOIN_LEFT);
        $select
            ->where
            ->equalTo(DatabaseService::USER_WALLET_TBL . '.user_id', $id);
        $result = $this->sql->prepareStatementForSqlObject($select)->execute();
        if ($result->count())
            return $result->current();
        return [];
    }

    public function getUserWalletHistory($id)
    {
        $select = $this->sql->select(DatabaseService::WALLET_HISTORY_TBL);
        $select
            ->where
            ->equalTo('user_id', $id);
        $select->order('id DESC');

        $result = $this->sql->prepareStatementForSqlObject($select)->execute();

        $array = [];
        if ($result->count()) {
            foreach ($result as $key => $value) {
                $array[$key] = $value;
            }
        }

        return $array;
    }

    public function getUserWalletTransactions($id)
    {
        $select = $this->sql->select(DatabaseService::WALLET_TRANS_TBL);
        $select
            ->where
            ->equalTo('user_id', $id);
        $select->order('id DESC');

        $result = $this->sql->prepareStatementForSqlObject($select)->execute();

        $array = [];
        if ($result->count()) {
            foreach ($result as $key => $value) {
                $array[$key] = $value;
            }
        }

        return $array;
    }


    public function getUserSettings($id)
    {
        $select = $this->sql->select(DatabaseService::USER_SETTINGS_TBL);
        $select
            ->where
            ->equalTo('user_id', $id);
        $result = $this->sql->prepareStatementForSqlObject($select)->execute();
        if ($result->count())
            return $result->current();
        return [];
    }


    public function getUserGuardians($id)
    {
        $select = $this->sql->select(DatabaseService::GUARDIAN_TBL);
        $select
            ->where
            ->equalTo('user_id', $id);
        $result = $this->sql->prepareStatementForSqlObject($select)->execute();

        $array = [];
        if ($result->count()) {
            foreach ($result as $key => $value) {
                $array[$key] = $value;
            }
        }

        return $array;
    }

    public function getUserGrades($id)
    {
        $select = $this->sql->select(DatabaseService::GRADES_TBL);
        $select
            ->where
            ->equalTo('student_id', $id);
        $select->order('id DESC');

        $result = $this->sql->prepareStatementForSqlObject($select)->execute();

        $array = [];
        if ($result->count()) {
            foreach ($result as $key => $value) {
                $array[$key] = $value;
            }
        }

        return $array;
    }

    public function getUserSubjects($id)
    {
        $array = [];
        $user  = $this->getUserInfo($id);
        $level = '';
        if (count($user))
            $level = $user['level'];

        if ($level != '') {
            $select = $this->sql->select(DatabaseService::SUBJECT_TBL);
            $select
                ->where
                ->equalTo('level_id', $level);

            $result = $this->sql->prepareStatementForSqlObject($select)->execute();

            if ($result->count()) {
                foreach ($result as $key => $value) {
                    $array['subjects'][$key] = $value;
                }
            }
        }

        return $array;
    }


    public function getUserBills($id)
    {
        $select = $this->sql->select(DatabaseService::PAYMENT_USER_MAP_TBL);
        $select->join(DatabaseService::PAYMENT_PARTICULAR_TBL, DatabaseService::PAYMENT_PARTICULAR_TBL . '.id = ' . DatabaseService::PAYMENT_USER_MAP_TBL . '.payment_particular_id', ['particular_id' => 'id', 'particular_name' => 'name', 'amount'], $select::JOIN_LEFT);
        $select->join(DatabaseService::PAYMENT_SCHEME_TBL, DatabaseService::PAYMENT_SCHEME_TBL . '.id = ' . DatabaseService::PAYMENT_USER_MAP_TBL . '.payment_scheme_id', ['scheme_id' => 'id', 'scheme_name' => 'name'], $select::JOIN_LEFT);
        $select
            ->where
            ->equalTo('user_id', $id);
        $select->order('id DESC');

        $result = $this->sql->prepareStatementForSqlObject($select)->execute();

        $array = [];
        if ($result->count()) {
            foreach ($result as $key => $value) {
                $array[$key] = $value;
            }
        }

        return $array;
    }

    public function getUserSchedules($id)
    {
        $select = $this->sql->select(DatabaseService::DY_SCHEDULE_TBL);
        $select
            ->where
            ->equalTo('user_id', $id);
        $select->order('id ASC');

        $result = $this->sql->prepareStatementForSqlObject($select)->execute();

        $array = [];
        if ($result->count()) {
            foreach ($result as $key => $value) {
                $array[$key] = $value;
            }
        }

        return $array;
    }

    public function getUsersAttendanceRecords($id)
    {
        $select = $this->sql->select(DatabaseService::ATTENDANCE_TBL);
        $select
            ->where
            ->equalTo('user_id', $id);
        $select->order('id DESC');

        $result = $this->sql->prepareStatementForSqlObject($select)->execute();

        $array = [];
        if ($result->count()) {
            foreach ($result as $key => $value) {
                $array[$key] = $value;
            }
        }

        return $array;
    }


    public function getAllUserInfo($id)
    {
        return [
            'attendance'          => $this->getUsersAttendanceRecords($id),
            'basic'               => $this->getUserInfo($id),
            'canteen'             => $this->getUserCanteenInfo($id),
            'wallet'              => $this->getUserWallet($id),
            'wallet_history'      => $this->getUserWalletHistory($id),
            'wallet_transactions' => $this->getUserWalletTransactions($id),
            'settings'            => $this->getUserSettings($id),
            'guardians'           => $this->getUserGuardians($id),
            'grades'              => $this->getUserGrades($id),
            'subjects'            => $this->getUserSubjects($id),
            'bills'               => $this->getUserBills($id),
            'schedules'           => $this->getUserSchedules($id),
        ];
    }

    public function generateId($type, $schoolId, $rowCount, $limit = 5, $newYear = 0)
    {
        $year           = date('Y'); //year today
        $school_id_fill = str_pad($schoolId, 2, "0", STR_PAD_LEFT); //add zero before schoolId if it the length is less that limit

        if ($newYear)
            $rowCount = 1;
        else $rowCount = $rowCount + 1;

        $user_id_fill = str_pad($rowCount, $limit, "0", STR_PAD_LEFT); //add zero before studentId if it the length is less that limit
        $id           = $type . $school_id_fill . $year . $user_id_fill; //composer id

        return $id;
    }

    public function getExistingDataByColumnValue($column, $value, $column2, $value2)
    {
        $select = $this->sql->select(DatabaseService::USER_TABLE);
        $select
            ->where
            ->equalTo($column, $value)
            ->and
            ->notEqualTo($column2, $value2);

        return $this->sql->prepareStatementForSqlObject($select)->execute();
    }

    public function setDatabase($adapter)
    {
        $this->sql = new Sql($adapter);
    }

    public function createService(ServiceLocatorInterface $serviceLocator)
    {
        return parent::createService($serviceLocator); // TODO: Change the autogenerated stub
    }


}